// Background script for the Kioju extension.  Listens for messages
// from the popup and performs network requests to share the current
// tab's URL with the user's Kioju server.

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'shareCurrentPage') {
    // Get sharing options from request
    const options = request.options || {};
    const isPublic = options.isPublic || false;
    const captureDescription = options.captureDescription !== false; // default to true
    
    // Query the active tab to get its URL
    chrome.tabs.query({ active: true, currentWindow: true }, function (tabs) {
      const currentTab = tabs && tabs.length > 0 ? tabs[0] : null;
      if (!currentTab || !currentTab.url) {
        sendResponse({ message: 'Unable to get current tab URL', success: false });
        return;
      }
      const url = currentTab.url;
      const title = currentTab.title || '';
      // Retrieve settings from storage
      chrome.storage.sync.get(['serverUrl', 'apiKey'], function (data) {
        const serverUrl = data.serverUrl;
        const apiKey = data.apiKey;
        if (!serverUrl || !apiKey) {
          sendResponse({ message: 'Please set server URL and API key in options', success: false });
          return;
        }
        // Build a POST request.  Send the API key in a custom header and
        // include the URL in the request body.  Use x-www-form-urlencoded
        // encoding so that PHP can read the parameters via $_POST.
        try {
          const endpoint = new URL(serverUrl);
          const params = new URLSearchParams();
          params.set('action', 'add');
          params.set('url', url);
          if (title) {
            params.set('title', title);
          }
          // Add new options
          params.set('is_private', isPublic ? '0' : '1'); // Convert to string, invert logic (isPublic -> is_private)
          params.set('capture_description', captureDescription ? '1' : '0');
          
          fetch(endpoint.toString(), {
            method: 'POST',
            headers: {
              'Content-Type': 'application/x-www-form-urlencoded',
              'X-Api-Key': apiKey
            },
            body: params.toString()
          })
            .then((resp) => resp.json().then((json) => ({ status: resp.status, json })))
            .then(({ status, json }) => {
              if (status === 201 && json.success) {
                sendResponse({ message: 'Link shared successfully', success: true });
              } else if (status === 429) {
                // Rate limit exceeded - provide helpful information
                const rateLimitInfo = json.rate_limit;
                let message = json.message || 'Rate limit exceeded';

                if (rateLimitInfo) {
                  message = `Rate limit: ${rateLimitInfo.current}/${rateLimitInfo.limit} submissions used. Try again in ${rateLimitInfo.reset_in_minutes || 'about 60'} minutes`;
                }

                sendResponse({ message: message, success: false, rateLimited: true });
              } else {
                sendResponse({ message: 'Error: ' + (json.message || 'Unknown error'), success: false });
              }
            })
            .catch((err) => {
              sendResponse({ message: 'Error sending request: ' + err.message, success: false });
            });
        } catch (e) {
          sendResponse({ message: 'Invalid server URL', success: false });
        }
      });
      // Indicate that we will respond asynchronously
      return true;
    });
    // Indicate asynchronous response
    return true;
  }
});